/*
* 	iterate Copyright (C) 2017-2025 Jérémy Pagès <jer.pages@tutanota.com>
*	
*	This file is part of iterate.
*
*	iterate is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	iterate is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with iterate.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <complex.h>
#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "colors.h"
#include "drawpng.h"
#include "mandelbrot.h"

#define VERSION "0.2"

void print_version()
{
	printf("Iterate, version %s\n", VERSION);
	exit(EXIT_SUCCESS);
}

void usage(int status)
{
	if (status == EXIT_FAILURE) {
		fprintf(stderr, "iterate : invalid option\n");
		fprintf(stderr, "Try 'iterate --help' for more information\n");
	}
	else {
		printf("Usage : iterate [OPTION] [PARAMETER(S)] ...\n");
		printf("Options :\n");
		printf("	-c		Set the image center affix\n");
		printf("	-m		Set the magnification\n");
		printf("	-n		Set the maximum number of iterations\n");
		printf("	-o		Save the computed image in the given file\n");
		printf("	-p		Set the color palette\n");
		printf("	-r		Set the resolution\n");
		printf("	--colors-period		Set the number of iterations after which colors are repeated\n");
		printf("	--help		Display this help and exit\n");
		printf("	--version	Output version information and exit\n");
	}
	exit(status);
}

int main(int argc, char **argv)
{
	double center_re = 0.0, center_im = 0.0;
	int width = 800;
	int height = 800;
	double magn = 1.0;
	int iter = 100;
	char *output_filename = "mandelbrot.png";
	char *palette = "";
	int colors_count = iter;

	int opt = 0;
	static struct option long_options[] = {
		{"colors-period",	required_argument, 0, 0},
		{"help",	no_argument,	0,	0},
		{"version",	no_argument,	0,	0},
		{0,	0,	0,	0}
	};

	while (opt != -1) {
		int option_index = 0;

		opt = getopt_long(argc, argv, "c:m:n:o:p:r:", long_options, &option_index);
		if (opt == 0) {
			if (option_index == 0)
				colors_count = strtol(optarg, NULL, 10);
			if (option_index == 1)
				usage(EXIT_SUCCESS);
			else if (option_index == 2)
				print_version();
			break;
		}
		if (opt == 'c') {
			center_re = strtod(argv[optind-1], NULL);
			center_im = strtod(argv[optind], NULL);
		}
		if (opt == 'm') {
			magn = strtod(optarg, NULL);
		}
		if (opt == 'n') {
			iter = strtol(optarg, NULL, 10);
		}
		if (opt == 'o') {
			output_filename = optarg;
		}
		if (opt == 'p') {
			palette = optarg;
		}
		if (opt == 'r') {
			width = strtol(strtok(optarg, "x"), NULL, 10);
			height = strtol(strtok(NULL, "x"), NULL, 10);
		}
		if (opt == '?') {
			usage(EXIT_FAILURE);
		}
	}

	double complex center_affix = center_re + I * center_im;

	unsigned int *buffer = calc_mandelbrot(center_affix, magn, iter, width, height);
	ColorList colors = compute_colors(palette, colors_count);
	write_png(output_filename, width, height, buffer, colors);

	free(buffer);

	return 0;
}
